<?php
require_once '../config/config.php';
require_once '../config/database.php';

// Cek login dan role (root dan administrator diizinkan)
if (!isLoggedIn() || !(hasRole('administrator') || hasRole('root'))) {
redirect('/login');
}

$error = '';
$success = '';

// Proses tambah user
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $username = sanitize($_POST['username']);
    $password = $_POST['password'];
    $nama_lengkap = sanitize($_POST['nama_lengkap']);
    $email = sanitize($_POST['email']);
    $role = sanitize($_POST['role']);
    $currentRole = $_SESSION['user_role'] ?? '';
    // Fallback: jika role kosong namun username adalah 'root', perlakukan sebagai root
    if (empty($currentRole) && (isset($_SESSION['user_username']) && $_SESSION['user_username'] === 'root')) {
        $currentRole = 'root';
    }
    
    // Validasi input
    if (empty($username) || empty($password) || empty($nama_lengkap) || empty($email) || empty($role)) {
        $error = 'Semua field harus diisi';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Format email tidak valid';
    } elseif (!in_array($role, ['root', 'administrator', 'verifikator', 'pendidik'])) {
        $error = 'Role tidak valid';
    } elseif ($currentRole === 'administrator' && in_array($role, ['administrator','root'])) {
        $error = 'Administrator tidak dapat menambahkan akun administrator atau root';
    } else {
        // Cek username dan email sudah ada atau belum
        $stmt = $conn->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
        $stmt->bind_param("ss", $username, $email);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $error = 'Username atau email sudah digunakan';
        } else {
            // Hash password
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            
            // Insert user baru
            $stmt = $conn->prepare("INSERT INTO users (username, password, nama_lengkap, email, role) VALUES (?, ?, ?, ?, ?)");
            $stmt->bind_param("sssss", $username, $hashed_password, $nama_lengkap, $email, $role);
            
            if ($stmt->execute()) {
                $success = 'Pengguna berhasil ditambahkan';
                
                // Log aktivitas
                $user_id = $_SESSION['user_id'];
                $aktivitas = "Menambahkan pengguna baru: $username";
                $ip = $_SERVER['REMOTE_ADDR'];
                $stmt = $conn->prepare("INSERT INTO log_aktivitas (user_id, aktivitas, ip_address) VALUES (?, ?, ?)");
                $stmt->bind_param("iss", $user_id, $aktivitas, $ip);
                $stmt->execute();
                
                // Reset form
                $username = $nama_lengkap = $email = $role = '';
            } else {
                $error = 'Gagal menambahkan pengguna: ' . $conn->error;
            }
        }
    }
}

// Include header
$pageTitle = "Tambah Pengguna";
include '../includes/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Tambah Pengguna Baru</h2>
<a href="<?= BASE_URL ?>/pages/users" class="btn btn-secondary">
        <i class="fas fa-arrow-left"></i> Kembali
    </a>
</div>

<div class="card">
    <div class="card-body">
        <?php if (!empty($error)): ?>
            <div class="alert alert-danger"><?= $error ?></div>
        <?php endif; ?>
        
        <?php if (!empty($success)): ?>
            <div class="alert alert-success"><?= $success ?></div>
        <?php endif; ?>
        
        <form method="POST" action="">
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="username" class="form-label">Username</label>
                    <input type="text" class="form-control" id="username" name="username" value="<?= $username ?? '' ?>" required>
                </div>
                <div class="col-md-6">
                    <label for="password" class="form-label">Password</label>
                    <input type="password" class="form-control" id="password" name="password" required>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="nama_lengkap" class="form-label">Nama Lengkap</label>
                    <input type="text" class="form-control" id="nama_lengkap" name="nama_lengkap" value="<?= $nama_lengkap ?? '' ?>" required>
                </div>
                <div class="col-md-6">
                    <label for="email" class="form-label">Email</label>
                    <input type="email" class="form-control" id="email" name="email" value="<?= $email ?? '' ?>" required>
                </div>
            </div>
            
            <div class="mb-3">
                <label for="role" class="form-label">Role</label>
                <select class="form-select" id="role" name="role" required>
                    <option value="" disabled selected>Pilih Role</option>
                    <?php 
                    $currentRole = $_SESSION['user_role'] ?? '';
                    // Fallback untuk root berdasarkan username
                    if (empty($currentRole) && (isset($_SESSION['user_username']) && $_SESSION['user_username'] === 'root')) {
                        $currentRole = 'root';
                    }
                    if ($currentRole === 'root') {
                        // Root dapat menambahkan Administrator, Verifikator, Pendidik
                        $roleOptions = [
                            'administrator' => 'Administrator',
                            'verifikator' => 'Verifikator',
                            'pendidik' => 'Pendidik'
                        ];
                    } else {
                        // Administrator hanya dapat menambahkan Verifikator dan Pendidik
                        $roleOptions = [
                            'verifikator' => 'Verifikator',
                            'pendidik' => 'Pendidik'
                        ];
                    }
                    foreach ($roleOptions as $val => $label) {
                        $selected = (isset($role) && $role == $val) ? 'selected' : '';
                        echo '<option value="'.$val.'" '.$selected.'>'.$label.'</option>';
                    }
                    ?>
                </select>
            </div>
            
            <div class="d-grid gap-2">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i> Simpan
                </button>
            </div>
        </form>
    </div>
</div>

<?php include '../includes/footer.php'; ?>