<?php
require_once '../config/config.php';
require_once '../config/database.php';

// Hanya root dan administrator
if (!isLoggedIn() || !(hasRole('administrator') || hasRole('root'))) {
redirect('/login');
}

$error = '';
$success = '';
$report = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validasi CSRF
    $token = $_POST['csrf_token'] ?? '';
    if (!verify_csrf($token)) {
        $error = 'Sesi tidak valid. Silakan muat ulang halaman dan coba lagi.';
    } else {
    if (!isset($_FILES['csv']) || $_FILES['csv']['error'] !== UPLOAD_ERR_OK) {
        $error = 'File CSV tidak valid atau gagal diunggah';
    } else {
        $fileTmp = $_FILES['csv']['tmp_name'];
        $handle = fopen($fileTmp, 'r');
        if (!$handle) {
            $error = 'Tidak dapat membaca file CSV';
        } else {
            $currentRole = $_SESSION['user_role'] ?? '';
            $allowedRoles = ($currentRole === 'root') ? ['administrator','verifikator','pendidik'] : ['verifikator','pendidik'];

            // Baca header
            $header = fgetcsv($handle);
            $expected = ['username','password','nama_lengkap','email','role','status'];
            $isHeaderOk = $header && count($header) >= 6 && array_map('strtolower', $header) === $expected;
            if (!$isHeaderOk) {
                $error = 'Header CSV tidak sesuai. Gunakan template yang disediakan.';
                fclose($handle);
            } else {
                $created = 0; $skipped = 0;
                while (($row = fgetcsv($handle)) !== false) {
                    if (count($row) < 6) { $report[] = ['row'=>$row, 'status'=>'skip', 'reason'=>'Kolom tidak lengkap']; $skipped++; continue; }
                    list($username,$password,$nama,$email,$role,$status) = $row;
                    $username = trim($username);
                    $password = trim($password);
                    $nama = trim($nama);
                    $email = trim($email);
                    $role = strtolower(trim($role));
                    $status = trim($status) === '1' ? 1 : 0;

                    // Validasi dasar
                    if ($username === '' || $password === '' || $nama === '' || $email === '' || $role === '') {
                        $report[] = ['row'=>$row, 'status'=>'skip', 'reason'=>'Field wajib kosong']; $skipped++; continue;
                    }
                    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                        $report[] = ['row'=>$row, 'status'=>'skip', 'reason'=>'Email tidak valid']; $skipped++; continue;
                    }
                    if (!in_array($role, ['administrator','verifikator','pendidik','root'])) {
                        $report[] = ['row'=>$row, 'status'=>'skip', 'reason'=>'Role tidak dikenal']; $skipped++; continue;
                    }
                    if (!in_array($role, $allowedRoles)) {
                        $report[] = ['row'=>$row, 'status'=>'skip', 'reason'=>'Role tidak diizinkan untuk akun Anda']; $skipped++; continue;
                    }
                    // Cegah impor root melalui CSV
                    if ($role === 'root') {
                        $report[] = ['row'=>$row, 'status'=>'skip', 'reason'=>'Impor akun root tidak diizinkan']; $skipped++; continue;
                    }

                    // Cek duplikasi
                    $stmt = $conn->prepare('SELECT id FROM users WHERE username = ? OR email = ?');
                    $stmt->bind_param('ss', $username, $email);
                    $stmt->execute();
                    $res = $stmt->get_result();
                    if ($res && $res->num_rows > 0) {
                        $report[] = ['row'=>$row, 'status'=>'skip', 'reason'=>'Username atau email sudah ada']; $skipped++; continue;
                    }

                    // Simpan
                    $hashed = password_hash($password, PASSWORD_DEFAULT);
                    $stmt = $conn->prepare('INSERT INTO users (username,password,nama_lengkap,email,role,status) VALUES (?,?,?,?,?,?)');
                    $stmt->bind_param('sssssi', $username, $hashed, $nama, $email, $role, $status);
                    if ($stmt->execute()) {
                        $created++;
                    } else {
                        $report[] = ['row'=>$row, 'status'=>'skip', 'reason'=>'Gagal menyimpan: '.$conn->error]; $skipped++;
                    }
                }
                fclose($handle);

                $success = "Impor selesai: $created dibuat, $skipped dilewati";
                // Log aktivitas
                $admin_id = (int)$_SESSION['user_id'];
                $aktivitas = 'Impor pengguna via CSV';
                $detail = json_encode(['dibuat'=>$created,'dilewati'=>$skipped]);
                $stmt = $conn->prepare("INSERT INTO log_aktivitas (user_id, aktivitas, detail, ip_address) VALUES (?, ?, ?, ?)");
                $ip = $_SERVER['REMOTE_ADDR'] ?? '';
                $stmt->bind_param('isss', $admin_id, $aktivitas, $detail, $ip);
                $stmt->execute();
            }
        }
    }
}
}

$pageTitle = 'Impor Pengguna (CSV)';
include '../includes/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Impor Pengguna dari CSV</h2>
    <div>
        <a href="<?= BASE_URL ?>/assets/templates/user_import_template.csv" class="btn btn-outline-secondary me-2">
            <i class="fas fa-download"></i> Unduh Template CSV
        </a>
<a href="<?= BASE_URL ?>/pages/users" class="btn btn-secondary">
            <i class="fas fa-arrow-left"></i> Kembali
        </a>
    </div>
    
</div>

<div class="card">
    <div class="card-body">
        <?php if (!empty($error)): ?>
            <div class="alert alert-danger"><?= $error ?></div>
        <?php endif; ?>
        <?php if (!empty($success)): ?>
            <div class="alert alert-success"><?= $success ?></div>
        <?php endif; ?>

        <form method="POST" enctype="multipart/form-data" action="">
            <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
            <div class="mb-3">
                <label for="csv" class="form-label">Pilih File CSV</label>
                <input type="file" class="form-control" id="csv" name="csv" accept=".csv" required>
                <small class="text-muted">Gunakan format kolom: username,password,nama_lengkap,email,role,status</small>
            </div>
            <button type="submit" class="btn btn-primary"><i class="fas fa-upload"></i> Impor</button>
        </form>
    </div>
</div>

<?php if (!empty($report)): ?>
<div class="card mt-4">
    <div class="card-header">Ringkasan Baris Dilewati</div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-sm">
                <thead>
                    <tr>
                        <th>Username</th>
                        <th>Email</th>
                        <th>Role</th>
                        <th>Alasan</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($report as $r): ?>
                        <tr>
                            <td><?= htmlspecialchars($r['row'][0] ?? '') ?></td>
                            <td><?= htmlspecialchars($r['row'][3] ?? '') ?></td>
                            <td><?= htmlspecialchars($r['row'][4] ?? '') ?></td>
                            <td><?= htmlspecialchars($r['reason'] ?? '-') ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
<?php endif; ?>

<?php include '../includes/footer.php'; ?>