<?php
require_once '../config/config.php';
require_once '../config/database.php';

if (!isLoggedIn()) {
redirect('/login');
}

$user_id = (int)$_SESSION['user_id'];
$stmt = $conn->prepare("SELECT id, username, nama_lengkap, email, role, status, created_at, updated_at FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
if (!$result || $result->num_rows === 0) {
    setAlert('Akun tidak ditemukan', 'danger');
redirect('/dashboard');
}
$user = $result->fetch_assoc();

// Proses ubah password akun
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    $token = $_POST['csrf_token'] ?? '';
    if (!verify_csrf($token)) {
        setAlert('Sesi tidak valid. Silakan muat ulang halaman.', 'danger');
        redirect('/pages/profile');
        exit;
    }

    $current = $_POST['current_password'] ?? '';
    $new = $_POST['new_password'] ?? '';
    $confirm = $_POST['confirm_password'] ?? '';

    if ($current === '' || $new === '' || $confirm === '') {
        setAlert('Semua field wajib diisi.', 'danger');
        redirect('/pages/profile');
        exit;
    }
    if ($new !== $confirm) {
        setAlert('Konfirmasi password tidak cocok.', 'danger');
        redirect('/pages/profile');
        exit;
    }
    if (strlen($new) < 8) {
        setAlert('Password baru minimal 8 karakter.', 'danger');
        redirect('/pages/profile');
        exit;
    }

    // Ambil hash password saat ini
    $stmtPw = $conn->prepare("SELECT password FROM users WHERE id = ? LIMIT 1");
    $stmtPw->bind_param("i", $user_id);
    $stmtPw->execute();
    $resPw = $stmtPw->get_result();
    if (!$resPw || $resPw->num_rows !== 1) {
        $stmtPw->close();
        setAlert('Akun tidak ditemukan.', 'danger');
        redirect('/pages/profile');
        exit;
    }
    $rowPw = $resPw->fetch_assoc();
    $stmtPw->close();

    if (!password_verify($current, $rowPw['password'])) {
        setAlert('Password saat ini salah.', 'danger');
        redirect('/pages/profile');
        exit;
    }

    // Update password
    $newHash = password_hash($new, PASSWORD_DEFAULT);
    $upd = $conn->prepare("UPDATE users SET password = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?");
    $upd->bind_param("si", $newHash, $user_id);
    if ($upd->execute()) {
        logActivity('Ubah password akun', 'Password akun berhasil diubah');
        setAlert('Password berhasil diubah.', 'success');
    } else {
        setAlert('Gagal mengubah password: ' . $conn->error, 'danger');
    }
    $upd->close();
    redirect('/pages/profile');
    exit;
}

$pageTitle = 'Profil Akun';
include '../includes/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Profil Akun</h2>
<a href="<?= BASE_URL ?>/dashboard" class="btn btn-secondary">
        <i class="fas fa-arrow-left"></i> Kembali
    </a>
</div>

<div class="card">
    <div class="card-body">
        <div class="row mb-3">
            <div class="col-md-6">
                <label class="form-label">Nama Lengkap</label>
                <input type="text" class="form-control" value="<?= htmlspecialchars($user['nama_lengkap']) ?>" readonly>
            </div>
            <div class="col-md-6">
                <label class="form-label">Email</label>
                <input type="text" class="form-control" value="<?= htmlspecialchars($user['email']) ?>" readonly>
            </div>
        </div>
        <div class="row mb-3">
            <div class="col-md-6">
                <label class="form-label">Username</label>
                <input type="text" class="form-control" value="<?= htmlspecialchars($user['username']) ?>" readonly>
            </div>
            <div class="col-md-3">
                <label class="form-label">Role</label>
                <input type="text" class="form-control" value="<?= $user['role'] == 'root' ? 'Root (Super Administrator)' : ucfirst($user['role']) ?>" readonly>
            </div>
            <div class="col-md-3">
                <label class="form-label">Status</label>
                <input type="text" class="form-control" value="<?= $user['status'] == 1 ? 'Aktif' : 'Nonaktif' ?>" readonly>
            </div>
        </div>
        <div class="row mb-3">
            <div class="col-md-6">
                <label class="form-label">Dibuat</label>
                <input type="text" class="form-control" value="<?= date('d/m/Y H:i', strtotime($user['created_at'])) ?>" readonly>
            </div>
            <div class="col-md-6">
                <label class="form-label">Diperbarui</label>
                <input type="text" class="form-control" value="<?= date('d/m/Y H:i', strtotime($user['updated_at'])) ?>" readonly>
            </div>
        </div>
    </div>
</div>

<div class="card mt-4">
    <div class="card-header"><i class="fas fa-key"></i> Ubah Password Akun</div>
    <div class="card-body">
        <form method="POST" action="">
            <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
            <input type="hidden" name="change_password" value="1">
            <div class="row mb-3">
                <div class="col-md-4">
                    <label class="form-label">Password Saat Ini</label>
                    <input type="password" class="form-control" name="current_password" required autocomplete="current-password">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Password Baru</label>
                    <input type="password" class="form-control" name="new_password" required autocomplete="new-password">
                    <small class="text-muted">Minimal 8 karakter.</small>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Konfirmasi Password Baru</label>
                    <input type="password" class="form-control" name="confirm_password" required autocomplete="new-password">
                </div>
            </div>
            <button type="submit" class="btn btn-primary"><i class="fas fa-save"></i> Ubah Password</button>
        </form>
    </div>
</div>

<?php include '../includes/footer.php'; ?>