<?php
// Konfigurasi Aplikasi
// Penguatan sesi untuk lingkungan publik
$secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
ini_set('session.use_strict_mode', 1);
ini_set('session.cookie_httponly', 1);
if ($secure) {
    ini_set('session.cookie_secure', 1);
}
if (PHP_VERSION_ID >= 70300) {
    session_set_cookie_params([
        'lifetime' => 0,
        'path' => '/',
        'domain' => '',
        'secure' => $secure,
        'httponly' => true,
        'samesite' => 'Lax'
    ]);
}
session_start();
// Set timezone aplikasi agar konsisten dengan MySQL/server
// Sesuaikan jika server Anda memakai zona waktu lain
date_default_timezone_set('Asia/Jakarta');

// Base URL: dukung override via environment untuk produksi (hanya jika URL valid)
$override = getenv('APP_BASE_URL') ?: getenv('APP_URL');
if ($override && preg_match('#^https?://#', $override)) {
    define('BASE_URL', rtrim($override, '/'));
} else {
    // Base URL dinamis: gunakan skema (http/https), host, dan base path relatif terhadap DOCUMENT_ROOT
    $scheme = $secure ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $appRoot = str_replace('\\', '/', dirname(__DIR__));
    $docRoot = str_replace('\\', '/', $_SERVER['DOCUMENT_ROOT'] ?? '');
    $basePath = '';
    if ($docRoot !== '') {
        $docReal = str_replace('\\', '/', realpath($docRoot) ?: $docRoot);
        $appReal = str_replace('\\', '/', realpath($appRoot) ?: $appRoot);
        if (strpos($appReal, $docReal) === 0) {
            $suffix = substr($appReal, strlen($docReal));
            $suffix = rtrim($suffix, '/');
            $basePath = ($suffix !== '') ? ('/' . ltrim($suffix, '/')) : '';
        }
    }
    // Fallback jika DOCUMENT_ROOT tidak tersedia: deteksi via REQUEST_URI untuk subfolder yang mengandung nama app
    if ($basePath === '') {
        $dirName = basename(dirname(__DIR__));
        $uriRaw = $_SERVER['REQUEST_URI'] ?? '';
        $uriPath = parse_url($uriRaw, PHP_URL_PATH) ?: '';
        $hasAppSegment = (strpos($uriPath, '/' . $dirName . '/') !== false) || preg_match('#^/' . preg_quote($dirName, '#') . '($|/)#', $uriPath);
        if ($hasAppSegment) {
            $basePath = '/' . $dirName;
        }
    }
    define('BASE_URL', $scheme . '://' . $host . $basePath);
}

// Direktori Upload
define('UPLOAD_DIR', __DIR__ . '/../uploads/');

// Ukuran maksimum file (1MB)
define('MAX_FILE_SIZE', 1 * 1024 * 1024);

// Tipe file yang diizinkan
define('ALLOWED_TYPES', [
    'application/pdf',
    'application/msword',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'application/vnd.ms-excel',
    'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    'image/jpeg',
    'image/png'
]);

// Fungsi untuk redirect
function redirect($url) {
    $target = (preg_match('#^https?://#', $url)) ? $url : (BASE_URL . $url);
    header("Location: " . $target);
    exit();
}

// Helper: render pagination kompak dengan halaman awal/akhir dan ellipsis
// Pemakaian: echo render_pagination($currentPage, $totalPages, function($p){ return BASE_URL.'/pages/activity?page='.$p; });
function render_pagination($currentPage, $totalPages, $buildLink, $ariaLabel = 'Navigasi halaman') {
    $currentPage = (int)$currentPage;
    $totalPages = (int)$totalPages;
    if ($totalPages <= 1) { return ''; }

    $prevPage = max(1, $currentPage - 1);
    $nextPage = min($totalPages, $currentPage + 1);

    // Konstruksi daftar halaman: selalu tampilkan 1, window sekitar current, dan last
    $window = 2; // jumlah halaman di kiri/kanan current
    $pages = [];
    $pages[] = 1;
    $start = max(2, $currentPage - $window);
    $end   = min($totalPages - 1, $currentPage + $window);
    if ($start > 2) { $pages[] = '...'; }
    for ($p = $start; $p <= $end; $p++) { $pages[] = $p; }
    if ($end < $totalPages - 1) { $pages[] = '...'; }
    if ($totalPages > 1) { $pages[] = $totalPages; }

    // Output HTML Bootstrap
    $html = '<nav aria-label="'.htmlspecialchars($ariaLabel).'">';
    $html .= '<ul class="pagination justify-content-center">';

    // Prev
    $disabledPrev = ($currentPage <= 1) ? ' disabled' : '';
    $html .= '<li class="page-item'.$disabledPrev.'">';
    $html .= '<a class="page-link" href="'.htmlspecialchars($buildLink($prevPage)).'" tabindex="'.($disabledPrev?'-1':'0').'">Sebelumnya</a>';
    $html .= '</li>';

    // Pages + ellipsis
    foreach ($pages as $item) {
        if ($item === '...') {
            $html .= '<li class="page-item disabled"><span class="page-link">&hellip;</span></li>';
        } else {
            $active = ($item === $currentPage) ? ' active' : '';
            $html .= '<li class="page-item'.$active.'">';
            $html .= '<a class="page-link" href="'.htmlspecialchars($buildLink($item)).'">'.$item.'</a>';
            $html .= '</li>';
        }
    }

    // Next
    $disabledNext = ($currentPage >= $totalPages) ? ' disabled' : '';
    $html .= '<li class="page-item'.$disabledNext.'">';
    $html .= '<a class="page-link" href="'.htmlspecialchars($buildLink($nextPage)).'" tabindex="'.($disabledNext?'-1':'0').'">Berikutnya</a>';
    $html .= '</li>';

    $html .= '</ul>';
    $html .= '</nav>';
    return $html;
}

// Enforce single-device login: ensure DB has session columns and validate token
function ensureUserSessionColumns() {
    global $conn;
    if (!$conn) return;
    // Tambah kolom session_token dan session_token_issued_at jika belum ada
    $colTok = $conn->query("SHOW COLUMNS FROM users LIKE 'session_token'");
    if (!$colTok || $colTok->num_rows === 0) {
        @$conn->query("ALTER TABLE users ADD COLUMN session_token VARCHAR(64) NULL AFTER last_login");
    }
    $colTokAt = $conn->query("SHOW COLUMNS FROM users LIKE 'session_token_issued_at'");
    if (!$colTokAt || $colTokAt->num_rows === 0) {
        @$conn->query("ALTER TABLE users ADD COLUMN session_token_issued_at TIMESTAMP NULL DEFAULT NULL AFTER session_token");
    }
}

function setUserSessionToken($userId) {
    global $conn;
    if (!$conn) return null;
    ensureUserSessionColumns();
    $token = bin2hex(random_bytes(32));
    $stmt = $conn->prepare("UPDATE users SET session_token = ?, session_token_issued_at = CURRENT_TIMESTAMP WHERE id = ?");
    $stmt->bind_param('si', $token, $userId);
    $stmt->execute();
    $stmt->close();
    $_SESSION['session_token'] = $token;
    return $token;
}

// Fungsi untuk cek login dengan validasi token sesi tunggal
function isLoggedIn() {
    if (!isset($_SESSION['user_id'])) return false;
    if (!isset($_SESSION['session_token'])) return false;
    global $conn;
    if (!$conn) return true; // fallback jika koneksi belum siap
    ensureUserSessionColumns();
    $uid = (int)$_SESSION['user_id'];
    $stmt = $conn->prepare("SELECT session_token FROM users WHERE id = ?");
    $stmt->bind_param('i', $uid);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res->fetch_assoc();
    $stmt->close();
    $dbToken = $row['session_token'] ?? '';
    $ok = is_string($dbToken) && hash_equals($dbToken, $_SESSION['session_token']);
    if (!$ok) {
        // Token tidak cocok: paksa logout
        session_unset();
        session_destroy();
        return false;
    }
    return true;
}

// Fungsi untuk cek role dengan hirarki akses
function hasRole($role) {
    if (!isset($_SESSION['user_role'])) return false;
    $userRole = $_SESSION['user_role'];
    $username = isset($_SESSION['user_username']) ? $_SESSION['user_username'] : '';
    // Root / Super Administrator memiliki semua akses
    if ($userRole === 'root' || $userRole === 'super_administrator' || $username === 'root') return true;
    // Administrator memiliki akses administrator, verifikator, pendidik
    if ($userRole === 'administrator') {
        return in_array($role, ['administrator', 'verifikator', 'pendidik']);
    }
    // Verifikator memiliki akses verifikator dan pendidik
    if ($userRole === 'verifikator') {
        return in_array($role, ['verifikator', 'pendidik']);
    }
    // Pendidik hanya akses pendidik
    return $userRole === $role;
}

// Fungsi untuk sanitasi input
function sanitize($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

// CSRF utilities: generate and verify token stored in session
function csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function verify_csrf($token) {
    if (!isset($_SESSION['csrf_token'])) return false;
    return is_string($token) && hash_equals($_SESSION['csrf_token'], $token);
}

// Helper alert (menetapkan pesan ke session)
function setAlert($message, $type = 'info') {
    $_SESSION['alert'] = [
        'message' => $message,
        'type' => $type
    ];
}

// Helper log aktivitas
function logActivity($aktivitas, $detail = null) {
    if (!isset($_SESSION['user_id'])) return; // hanya log jika ada user
    $user_id = (int)$_SESSION['user_id'];
    $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    // Gunakan koneksi global
    global $conn;
    if (!$conn) return;
    $stmt = $conn->prepare("INSERT INTO log_aktivitas (user_id, aktivitas, detail, ip_address) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("isss", $user_id, $aktivitas, $detail, $ip);
    $stmt->execute();
}

// OTP konfigurasi dan helper
// Debug: tampilkan OTP di localhost untuk uji coba (matikan di produksi)
define('OTP_DEBUG_DISPLAY', false);

// Kebijakan OTP
define('OTP_TTL_MINUTES', 5); // masa berlaku maksimum 5 menit
define('OTP_RESEND_MIN_SECONDS', 0); // jeda minimum kirim ulang (izinkan kirim ulang segera)
define('OTP_MAX_RESENDS', 5); // batas kirim ulang
define('OTP_MAX_ATTEMPTS', 5); // batas percobaan

// Kunci enkripsi untuk menyimpan OTP secara terenkripsi agar bisa dikirim ulang
// Catatan: di produksi, simpan kunci ini di environment/secret storage.
define('OTP_SECRET_KEY', hash('sha256', 'change-this-key-in-production', true));

// Konfigurasi SMTP untuk pengiriman email (hanya Gmail)
define('SMTP_ENABLED', true); // aktifkan SMTP
define('SMTP_HOST', 'smtp.gmail.com');
define('SMTP_PORT', 587);
define('SMTP_SECURE', 'tls'); // STARTTLS
define('SMTP_USERNAME', 'reedha.adisty@gmail.com');
define('SMTP_PASSWORD', 'eicwjslnleujsnwm'); // App Password 16 digit
define('SMTP_FROM', 'reedha.adisty@gmail.com');
define('SMTP_FROM_NAME', 'e-File SMK Negeri 1 Salam');

// Pastikan tabel OTP tersedia; buat jika belum ada
function ensureOtpTable() {
    global $conn;
    if (!$conn) return;
    // Cek keberadaan tabel
    $check = $conn->query("SHOW TABLES LIKE 'user_otp'");
    if ($check && $check->num_rows > 0) {
        // Tambahkan kolom terenkripsi jika belum ada
        $colEnc = $conn->query("SHOW COLUMNS FROM user_otp LIKE 'otp_enc'");
        if (!$colEnc || $colEnc->num_rows === 0) {
            @$conn->query("ALTER TABLE user_otp ADD COLUMN otp_enc VARCHAR(128) NULL AFTER otp_hash");
        }
        $colIv = $conn->query("SHOW COLUMNS FROM user_otp LIKE 'otp_iv'");
        if (!$colIv || $colIv->num_rows === 0) {
            @$conn->query("ALTER TABLE user_otp ADD COLUMN otp_iv VARCHAR(32) NULL AFTER otp_enc");
        }
        $colTag = $conn->query("SHOW COLUMNS FROM user_otp LIKE 'otp_tag'");
        if (!$colTag || $colTag->num_rows === 0) {
            @$conn->query("ALTER TABLE user_otp ADD COLUMN otp_tag VARCHAR(32) NULL AFTER otp_iv");
        }
        // Pastikan kolom created_at tersedia untuk penegakan TTL 5 menit
        $colCreated = $conn->query("SHOW COLUMNS FROM user_otp LIKE 'created_at'");
        if (!$colCreated || $colCreated->num_rows === 0) {
            // Tambah kolom created_at dengan default CURRENT_TIMESTAMP
            @$conn->query("ALTER TABLE user_otp ADD COLUMN created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP AFTER resend_count");
            // Pengisian awal aman untuk SQL mode NO_ZERO_DATE: prioritaskan last_sent_at, lalu expires_at - TTL, jika keduanya tidak ada biarkan nilai default
            $ttl = (int)OTP_TTL_MINUTES;
            @$conn->query("UPDATE user_otp SET created_at = CASE WHEN last_sent_at IS NOT NULL THEN last_sent_at WHEN expires_at IS NOT NULL THEN DATE_SUB(expires_at, INTERVAL $ttl MINUTE) ELSE created_at END WHERE (last_sent_at IS NOT NULL OR expires_at IS NOT NULL)");
        }
        // Tambahkan kolom expires_at jika belum ada
        $colExpires = $conn->query("SHOW COLUMNS FROM user_otp LIKE 'expires_at'");
        if (!$colExpires || $colExpires->num_rows === 0) {
            @$conn->query("ALTER TABLE user_otp ADD COLUMN expires_at TIMESTAMP NULL DEFAULT NULL AFTER otp_tag");
            $ttl = (int)OTP_TTL_MINUTES;
            @$conn->query("UPDATE user_otp SET expires_at = DATE_ADD(COALESCE(created_at, last_sent_at, NOW()), INTERVAL $ttl MINUTE) WHERE expires_at IS NULL OR expires_at = '0000-00-00 00:00:00'");
            // Tambahkan index untuk expires jika belum ada
            $idxExp = $conn->query("SHOW INDEX FROM user_otp WHERE Key_name = 'idx_user_otp_expires'");
            if (!$idxExp || $idxExp->num_rows === 0) {
                @$conn->query("ALTER TABLE user_otp ADD KEY idx_user_otp_expires (expires_at)");
            }
        }
        // Tambahkan kolom attempts jika belum ada
        $colAttempts = $conn->query("SHOW COLUMNS FROM user_otp LIKE 'attempts'");
        if (!$colAttempts || $colAttempts->num_rows === 0) {
            @$conn->query("ALTER TABLE user_otp ADD COLUMN attempts INT DEFAULT 0 AFTER expires_at");
        }
        // Pastikan last_sent_at dan resend_count ada
        $colLastSent = $conn->query("SHOW COLUMNS FROM user_otp LIKE 'last_sent_at'");
        if (!$colLastSent || $colLastSent->num_rows === 0) {
            @$conn->query("ALTER TABLE user_otp ADD COLUMN last_sent_at TIMESTAMP NULL DEFAULT NULL AFTER attempts");
        }
        $colResend = $conn->query("SHOW COLUMNS FROM user_otp LIKE 'resend_count'");
        if (!$colResend || $colResend->num_rows === 0) {
            @$conn->query("ALTER TABLE user_otp ADD COLUMN resend_count INT DEFAULT 0 AFTER last_sent_at");
        }
        return; // sudah ada (dan kolom dilengkapi)
    }
    // Buat tabel jika belum ada
    $createSql = "CREATE TABLE IF NOT EXISTS user_otp (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        otp_hash VARCHAR(255) NOT NULL,
        otp_enc VARCHAR(128) NULL,
        otp_iv VARCHAR(32) NULL,
        otp_tag VARCHAR(32) NULL,
        expires_at TIMESTAMP NOT NULL,
        attempts INT DEFAULT 0,
        last_sent_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
        resend_count INT DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        KEY idx_user_otp_user (user_id),
        KEY idx_user_otp_expires (expires_at)
    )";
    @$conn->query($createSql);
}

// Kirim email OTP (sederhana). Produksi sebaiknya gunakan SMTP terkonfigurasi.
function sendOtpEmail($toEmail, $code) {
    $subject = 'Kode OTP Login e-File';
    $message = "Kode OTP Anda: $code\r\nBerlaku " . OTP_TTL_MINUTES . " menit. Jangan bagikan kode ini.";
    // Jika SMTP diaktifkan, kirim via SMTP
    if (SMTP_ENABLED && !empty(SMTP_HOST)) {
        return smtp_send_mail($toEmail, $subject, $message);
    }
    // Fallback ke mail() jika SMTP tidak dikonfigurasi
    $headers = 'From: ' . (SMTP_FROM ?: ('no-reply@' . ($_SERVER['HTTP_HOST'] ?? 'localhost')));
    $ok = @mail($toEmail, $subject, $message, $headers);
    return $ok || OTP_DEBUG_DISPLAY;
}

// Pengiriman email via SMTP sederhana tanpa library eksternal
function smtp_send_mail($toEmail, $subject, $body) {
    $host = SMTP_HOST;
    $port = (int)SMTP_PORT;
    $secure = strtolower(SMTP_SECURE);
    $username = SMTP_USERNAME;
    $password = SMTP_PASSWORD;
    $from = SMTP_FROM ?: $username;
    $fromName = SMTP_FROM_NAME ?: 'No-Reply';

    $remote = ($secure === 'ssl' ? "ssl://$host" : $host);
    $errno = 0; $errstr = '';
    $fp = @stream_socket_client($remote . ':' . $port, $errno, $errstr, 30, STREAM_CLIENT_CONNECT);
    if (!$fp) { return false; }
    stream_set_timeout($fp, 30);

    $read = function() use ($fp) {
        $resp = '';
        while (($line = fgets($fp, 515)) !== false) {
            $resp .= $line;
            if (strlen($line) < 4) break;
            if ($line[3] === ' ') break; // baris terakhir kode status
        }
        return $resp;
    };
    $cmd = function($command) use ($fp, $read) {
        fwrite($fp, $command . "\r\n");
        return $read();
    };

    $resp = $read(); // 220
    if (strpos($resp, '220') !== 0) { fclose($fp); return false; }

    $ehloHost = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $resp = $cmd('EHLO ' . $ehloHost);
    if (strpos($resp, '250') !== 0) {
        // coba HELO
        $resp = $cmd('HELO ' . $ehloHost);
        if (strpos($resp, '250') !== 0) { fclose($fp); return false; }
    }

    // STARTTLS jika diminta
    if ($secure === 'tls') {
        $resp = $cmd('STARTTLS');
        if (strpos($resp, '220') !== 0) { fclose($fp); return false; }
        if (!stream_socket_enable_crypto($fp, true, STREAM_CRYPTO_METHOD_TLS_CLIENT)) { fclose($fp); return false; }
        // EHLO ulang setelah TLS
        $resp = $cmd('EHLO ' . $ehloHost);
        if (strpos($resp, '250') !== 0) { fclose($fp); return false; }
    }

    // AUTH LOGIN jika kredensial tersedia
    if (!empty($username)) {
        $resp = $cmd('AUTH LOGIN');
        if (strpos($resp, '334') !== 0) { fclose($fp); return false; }
        $resp = $cmd(base64_encode($username));
        if (strpos($resp, '334') !== 0) { fclose($fp); return false; }
        $resp = $cmd(base64_encode($password));
        if (strpos($resp, '235') !== 0) { fclose($fp); return false; }
    }

    // MAIL FROM
    $resp = $cmd('MAIL FROM: <' . $from . '>' );
    if (strpos($resp, '250') !== 0) { fclose($fp); return false; }
    // RCPT TO
    $resp = $cmd('RCPT TO: <' . $toEmail . '>' );
    if (strpos($resp, '250') !== 0 && strpos($resp, '251') !== 0) { fclose($fp); return false; }
    // DATA
    $resp = $cmd('DATA');
    if (strpos($resp, '354') !== 0) { fclose($fp); return false; }

    // Header dan body (Plain Text)
    $headers = '';
    $headers .= 'From: ' . encode_header($fromName) . ' <' . $from . ">\r\n";
    $headers .= 'To: <' . $toEmail . ">\r\n";
    $headers .= 'Subject: ' . encode_header($subject) . "\r\n";
    $headers .= "MIME-Version: 1.0\r\n";
    $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
    $headers .= "Content-Transfer-Encoding: 8bit\r\n";
    $msg = $headers . "\r\n" . $body . "\r\n";
    fwrite($fp, $msg . ".\r\n");
    $resp = $read();
    if (strpos($resp, '250') !== 0) { fclose($fp); return false; }

    $cmd('QUIT');
    fclose($fp);
    return true;
}

function encode_header($str) {
    // RFC 2047 encoding untuk UTF-8 subject/name
    if (preg_match('/[^\x20-\x7E]/', $str)) {
        return '=?UTF-8?B?' . base64_encode($str) . '?=';
    }
    return $str;
}

function createOtpForUser($userId) {
    global $conn;
    if (!$conn) return false;
    ensureOtpTable();
    // Cek apakah ada OTP aktif yang belum kedaluwarsa
    $sel = $conn->prepare('SELECT id, otp_hash, expires_at, otp_enc, otp_iv, otp_tag FROM user_otp WHERE user_id = ? ORDER BY id DESC LIMIT 1');
    $sel->bind_param('i', $userId);
    $sel->execute();
    $res = $sel->get_result();
    $row = $res->fetch_assoc();
    $sel->close();

    if ($row && strtotime($row['expires_at']) > time()) {
        // Kode masih berlaku: kirim ulang kode yang sama
        $code = null;
        if (!empty($row['otp_enc']) && !empty($row['otp_iv']) && !empty($row['otp_tag'])) {
            $iv = base64_decode($row['otp_iv']);
            $tag = base64_decode($row['otp_tag']);
            $enc = base64_decode($row['otp_enc']);
            $code = openssl_decrypt($enc, 'aes-256-gcm', OTP_SECRET_KEY, OPENSSL_RAW_DATA, $iv, $tag);
        }
        if ($code === null || $code === false) {
            // Jika decrypt gagal, buat baru (tetap patuhi TTL 10 menit)
            $code = (string)random_int(100000, 999999);
            $hash = password_hash($code, PASSWORD_DEFAULT);
            $iv = random_bytes(12);
            $enc = openssl_encrypt($code, 'aes-256-gcm', OTP_SECRET_KEY, OPENSSL_RAW_DATA, $iv, $tag);
            $upd = $conn->prepare('UPDATE user_otp SET otp_hash = ?, otp_enc = ?, otp_iv = ?, otp_tag = ?, expires_at = DATE_ADD(NOW(), INTERVAL ' . (int)OTP_TTL_MINUTES . ' MINUTE), attempts = 0, resend_count = 0, last_sent_at = CURRENT_TIMESTAMP WHERE id = ?');
            $encB64 = base64_encode($enc);
            $ivB64 = base64_encode($iv);
            $tagB64 = base64_encode($tag);
            $upd->bind_param('ssssi', $hash, $encB64, $ivB64, $tagB64, $row['id']);
            $upd->execute();
            $upd->close();
        }
        if (OTP_DEBUG_DISPLAY) { $_SESSION['otp_last_code'] = $code; }
        return $code;
    }

    // Jika tidak ada atau sudah kedaluwarsa, hapus entry lama dan buat kode baru
    if ($row) {
        $del = $conn->prepare('DELETE FROM user_otp WHERE id = ?');
        $del->bind_param('i', $row['id']);
        $del->execute();
        $del->close();
    }

    $code = (string)random_int(100000, 999999);
    $hash = password_hash($code, PASSWORD_DEFAULT);
    $iv = random_bytes(12);
    $enc = openssl_encrypt($code, 'aes-256-gcm', OTP_SECRET_KEY, OPENSSL_RAW_DATA, $iv, $tag);
    $encB64 = base64_encode($enc);
    $ivB64 = base64_encode($iv);
    $tagB64 = base64_encode($tag);

    $ins = $conn->prepare('INSERT INTO user_otp (user_id, otp_hash, otp_enc, otp_iv, otp_tag, expires_at) VALUES (?, ?, ?, ?, ?, DATE_ADD(NOW(), INTERVAL ' . (int)OTP_TTL_MINUTES . ' MINUTE))');
    $ins->bind_param('issss', $userId, $hash, $encB64, $ivB64, $tagB64);
    $ins->execute();
    $ins->close();

    if (OTP_DEBUG_DISPLAY) { $_SESSION['otp_last_code'] = $code; }
    return $code;
}

function canResendOtp($userId) {
    global $conn;
    if (!$conn) return false;
    ensureOtpTable();
    $stmt = $conn->prepare('SELECT last_sent_at, resend_count FROM user_otp WHERE user_id = ? ORDER BY id DESC LIMIT 1');
    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $res = $stmt->get_result();
    if ($row = $res->fetch_assoc()) {
        $last = strtotime($row['last_sent_at'] ?? '1970-01-01');
        $now = time();
        $gap = $now - $last;
        $count = (int)($row['resend_count'] ?? 0);
        return $gap >= OTP_RESEND_MIN_SECONDS && $count < OTP_MAX_RESENDS; // min jeda, batas resends
    }
    return true; // jika belum ada, izinkan
}

function markOtpSent($userId) {
    global $conn;
    if (!$conn) return;
    ensureOtpTable();
    $stmt = $conn->prepare('UPDATE user_otp SET last_sent_at = NOW(), resend_count = resend_count + 1 WHERE user_id = ?');
    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $stmt->close();
}

function verifyOtpCode($userId, $code) {
    global $conn;
    if (!$conn) return false;
    ensureOtpTable();
    $stmt = $conn->prepare('SELECT id, otp_hash, expires_at, attempts, created_at, last_sent_at FROM user_otp WHERE user_id = ? ORDER BY id DESC LIMIT 1');
    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res->fetch_assoc();
    $stmt->close();
    if (!$row) return false;
    $now = time();
    $ttlSeconds = (int)OTP_TTL_MINUTES * 60;
    // Pilih issuedAt yang valid (hindari tanggal nol/invalid)
    $created = $row['created_at'] ?? '';
    $lastSent = $row['last_sent_at'] ?? '';
    $expires = $row['expires_at'] ?? '';
    $isValidDt = function($dt) {
        return is_string($dt) && $dt !== '' && $dt !== '0000-00-00 00:00:00' && strtotime($dt) !== false;
    };
    if ($isValidDt($created)) {
        $issuedAt = $created;
    } elseif ($isValidDt($lastSent)) {
        $issuedAt = $lastSent;
    } else {
        $issuedAt = $expires; // fallback terakhir
    }
    $issuedTs = strtotime($issuedAt);
    // Expired jika lewat TTL saat ini ATAU melewati expires_at yang tersimpan
    $expiredByTTL = ($issuedTs !== false) ? (($now - $issuedTs) > $ttlSeconds) : false;
    $expiresTs = strtotime($expires);
    $expiredByStored = ($expiresTs !== false) ? ($expiresTs < $now) : false; // jika tanggal invalid, jangan langsung dianggap expired
    $expired = $expiredByTTL || $expiredByStored;
    if ($expired) {
        // Hapus OTP yang kedaluwarsa untuk mencegah penggunaan kembali
        $del = $conn->prepare('DELETE FROM user_otp WHERE id = ?');
        $del->bind_param('i', $row['id']);
        $del->execute();
        $del->close();
        return false;
    }
    if ((int)$row['attempts'] >= OTP_MAX_ATTEMPTS) return false;
    $ok = password_verify($code, $row['otp_hash']);
    if ($ok) {
        // Hapus OTP setelah sukses
        $del = $conn->prepare('DELETE FROM user_otp WHERE id = ?');
        $del->bind_param('i', $row['id']);
        $del->execute();
        $del->close();
        return true;
    } else {
        // Tambah attempts
        $upd = $conn->prepare('UPDATE user_otp SET attempts = attempts + 1 WHERE id = ?');
        $upd->bind_param('i', $row['id']);
        $upd->execute();
        $upd->close();
        return false;
    }
}
?>