<?php
require_once '../config/config.php';
require_once '../config/database.php';

// Cek login dan role (pendidik, administrator, root). Verifikator tidak memiliki hak upload.
if (!isLoggedIn() || (!hasRole('pendidik') && !hasRole('administrator') && !hasRole('root'))) {
    setAlert('Anda tidak memiliki akses ke halaman ini', 'danger');
redirect('/login');
    exit;
}

// Ambil folder_id dari parameter URL
$folder_id = isset($_GET['folder_id']) ? (int)$_GET['folder_id'] : 0;
// Mode unggah ulang (replace) untuk dokumen draft
$isReplace = isset($_GET['replace']) && ($_GET['replace'] === '1' || $_GET['replace'] === 'true');

// Jika folder_id tidak valid, redirect ke halaman kategori
if ($folder_id <= 0) {
    setAlert('Silakan pilih folder terlebih dahulu', 'warning');
redirect('/pages/data_dinamis');
    exit;
}

// Ambil informasi folder
$folder_query = "SELECT f.id, f.nama_folder, k.id as kategori_id, k.nama_kategori, k.tipe 
                FROM folder f 
                JOIN kategori_dokumen k ON f.kategori_id = k.id 
                WHERE f.id = ?";
$stmt = $conn->prepare($folder_query);
$stmt->bind_param("i", $folder_id);
$stmt->execute();
$folder_result = $stmt->get_result();

if ($folder_result->num_rows == 0) {
    setAlert('Folder tidak ditemukan', 'danger');
redirect('/pages/data_dinamis');
    exit;
}

$folder = $folder_result->fetch_assoc();

// Proses upload dokumen
if (isset($_POST['upload'])) {
    // Validasi CSRF
    $token = $_POST['csrf_token'] ?? '';
    if (!verify_csrf($token)) {
        setAlert('Sesi tidak valid. Silakan muat ulang halaman dan coba lagi.', 'danger');
        redirect('/pages/upload?folder_id=' . $folder_id . ($isReplace ? '&replace=1' : ''));
        exit;
    }
    // Deskripsi dihilangkan sesuai permintaan
    $deskripsi = '';
    $kategori_id = (int)$folder['kategori_id'];
    // Tentukan pemilik dokumen
    $owner_id = (int)$_SESSION['user_id'];
    if ((hasRole('administrator') || hasRole('root')) && isset($_POST['owner_user_id'])) {
        $candidate = (int)$_POST['owner_user_id'];
        if ($candidate > 0) {
            // Validasi kandidat adalah pendidik aktif saja
            $stmtOwn = $conn->prepare("SELECT id FROM users WHERE id = ? AND role = 'pendidik' AND status = 1");
            $stmtOwn->bind_param("i", $candidate);
            $stmtOwn->execute();
            $resOwn = $stmtOwn->get_result();
            if ($resOwn && $resOwn->num_rows === 1) {
                $owner_id = $candidate;
            }
            $stmtOwn->close();
        }
    }
    
    // Ambil data pemilik (untuk penamaan file dan judul)
    $ownerData = null;
    $stmtOwner = $conn->prepare("SELECT username, nama_lengkap FROM users WHERE id = ?");
    $stmtOwner->bind_param("i", $owner_id);
    $stmtOwner->execute();
    $ownerRes = $stmtOwner->get_result();
    if ($ownerRes && $ownerRes->num_rows === 1) {
        $ownerData = $ownerRes->fetch_assoc();
    }
    $stmtOwner->close();

    // Validasi input: hanya file
    if (!isset($_FILES['dokumen']) || $_FILES['dokumen']['error'] != 0) {
        setAlert('File dokumen tidak valid', 'danger');
    } else {
        $file = $_FILES['dokumen'];
        $file_name = $file['name'];
        $file_size = $file['size'];
        $file_tmp = $file['tmp_name'];
        $file_type = $file['type'];
        // Judul/filename mengikuti format: NIP_Pendidik_Judul Folder
        // Gunakan nama folder sebagai komponen Judul Folder (slug)
        $folder_slug = isset($folder['nama_folder']) ? preg_replace('/\s+/', '_', trim($folder['nama_folder'])) : 'Folder';
        $folder_slug = preg_replace('/[^A-Za-z0-9_\-]/', '', $folder_slug);
        // Ambil NIP dari username pemilik (diasumsikan username = NIP), dan Nama dari nama_lengkap
        $nip = isset($ownerData['username']) ? preg_replace('/[^0-9]/', '', $ownerData['username']) : 'NIP';
        $nama = isset($ownerData['nama_lengkap']) ? preg_replace('/\s+/', '_', trim($ownerData['nama_lengkap'])) : 'Nama';
        $nama = preg_replace('/[^A-Za-z0-9_\-]/', '', $nama);
        $judulCombined = $nip . '_' . $nama . '_' . $folder_slug;
        $judul = sanitize($judulCombined);
        
        // Validasi ukuran file
        if ($file_size > MAX_FILE_SIZE) {
            setAlert('Ukuran file terlalu besar (maksimal ' . (MAX_FILE_SIZE / 1024 / 1024) . 'MB)', 'danger');
        } else {
            // Validasi tipe file: hanya PDF
            if ($file_type !== 'application/pdf') {
                setAlert('Tipe file tidak diizinkan. Hanya PDF.', 'danger');
            } else {
                // Buat direktori upload: pending verifikasi disimpan di uploads/verifikasi/<folder_id>
                $isPrivileged = (hasRole('administrator') || hasRole('root'));
                $upload_dir = $isPrivileged ? (UPLOAD_DIR . '/' . $folder_id) : (UPLOAD_DIR . '/verifikasi/' . $folder_id);
                if (!file_exists($upload_dir)) {
                    // Permission lebih ketat untuk produksi
                    mkdir($upload_dir, 0755, true);
                }
                
                // Gunakan nama file sesuai format standar (NIP_Pendidik_Judul Folder.pdf)
                $file_ext = pathinfo($file_name, PATHINFO_EXTENSION);
                $final_name = $judulCombined . '.pdf';
                $file_path = $upload_dir . '/' . $final_name;
                // Jika file dengan nama sama sudah ada, tambahkan akhiran numerik
                $counter = 1;
                while (file_exists($file_path)) {
                    $final_name = $judulCombined . '_' . $counter . '.pdf';
                    $file_path = $upload_dir . '/' . $final_name;
                    $counter++;
                }
                
                // Cek apakah sudah ada dokumen pada folder ini oleh pemilik (satu folder hanya boleh diupload sekali per pendidik)
                $stmtCheck = $conn->prepare("SELECT id, status, file_path FROM dokumen WHERE folder_id = ? AND user_id = ? ORDER BY id DESC LIMIT 1");
                $stmtCheck->bind_param("ii", $folder_id, $owner_id);
                $stmtCheck->execute();
                $existing = $stmtCheck->get_result()->fetch_assoc();
                $stmtCheck->close();

                // Jika ada dokumen dan bukan mode replace, batasi unggah
                if ($existing && !$isReplace) {
                    if ($existing['status'] !== 'draft') {
                        setAlert('Anda sudah mengunggah dokumen untuk folder ini. Tunggu verifikasi atau unggah ulang setelah menjadi draft.', 'warning');
                        redirect('/pages/documents?folder_id=' . $folder_id);
                        exit;
                    } else {
                        setAlert('Dokumen Anda berstatus draft. Silakan unggah ulang file perbaikan.', 'info');
                        redirect('/pages/upload?folder_id=' . $folder_id . '&replace=1');
                        exit;
                    }
                }

                // Validasi MIME menggunakan finfo untuk mencegah spoofing
                $finfo = new finfo(FILEINFO_MIME_TYPE);
                $detectedMime = $finfo->file($file_tmp);
                if ($detectedMime !== 'application/pdf') {
                    setAlert('Tipe file tidak diizinkan (deteksi MIME). Hanya PDF.', 'danger');
                } else if (move_uploaded_file($file_tmp, $file_path)) {
                    // Simpan informasi dokumen ke database
                    $relative_path = 'uploads/' . ($isPrivileged ? $folder_id : ('verifikasi/' . $folder_id)) . '/' . $final_name;
                    // Admin/root tidak perlu verifikasi
                    $initialStatus = (hasRole('administrator') || hasRole('root')) ? 'diverifikasi' : 'menunggu_verifikasi';
                    if ($isReplace && $existing) {
                        // Unggah ulang: update dokumen draft milik pendidik
                        $docId = (int)$existing['id'];
                        // Hapus file lama jika ada
                        if (!empty($existing['file_path'])) {
                            $oldAbs = dirname(__DIR__) . '/' . $existing['file_path'];
                            if (file_exists($oldAbs)) { @unlink($oldAbs); }
                        }
                        $stmt = $conn->prepare("UPDATE dokumen SET judul = ?, deskripsi = ?, file_path = ?, file_size = ?, file_type = ?, status = ? WHERE id = ?");
                        $stmt->bind_param("sssissi", $judul, $deskripsi, $relative_path, $file_size, $file_type, $initialStatus, $docId);
                    } else {
                        // Unggahan baru
                        $stmt = $conn->prepare("INSERT INTO dokumen (judul, deskripsi, file_path, file_size, file_type, kategori_id, folder_id, user_id, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
                        $stmt->bind_param("sssissiis", $judul, $deskripsi, $relative_path, $file_size, $file_type, $kategori_id, $folder_id, $owner_id, $initialStatus);
                    }
                    
                    if ($stmt->execute()) {
                        // Tangkap ID dokumen SEBELUM ada insert lain (log aktivitas)
                        $docIdAuto = ($isReplace && $existing) ? (int)$existing['id'] : (int)$conn->insert_id;

                        // Log aktivitas
                        logActivity('Mengupload dokumen baru: ' . $judul . ' ke folder ' . $folder['nama_folder']);

                        // Jika diverifikasi otomatis, catat ke tabel verifikasi
                        if ($initialStatus === 'diverifikasi') {
                            $verifikator_id = (int)$_SESSION['user_id'];
                            $catatan = 'Otomatis disetujui oleh ' . ($_SESSION['user_name'] ?? 'Admin');
                            $stmtV = $conn->prepare("INSERT INTO verifikasi (dokumen_id, verifikator_id, status, catatan) VALUES (?, ?, 'disetujui', ?)");
                            $stmtV->bind_param("iis", $docIdAuto, $verifikator_id, $catatan);
                            if ($stmtV->execute()) {
                                setAlert('Dokumen berhasil diupload dan langsung diverifikasi', 'success');
                            } else {
                                setAlert('Dokumen tersimpan, namun pencatatan verifikasi gagal: ' . $conn->error, 'warning');
                            }
                            $stmtV->close();
                        } else {
                            setAlert('Dokumen berhasil diupload dan menunggu verifikasi', 'success');
                        }
                        redirect('/pages/documents?folder_id=' . $folder_id);
                        exit;
                    } else {
                        setAlert('Gagal menyimpan informasi dokumen: ' . $conn->error, 'danger');
                    }
                    $stmt->close();
                } else {
                    setAlert('Gagal mengupload file', 'danger');
                }
            }
        }
    }
}

// Header
$pageTitle = "Upload Dokumen";
include '../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-12">
        <h2><i class="fas fa-upload"></i> Upload Dokumen</h2>
        <p class="text-muted">
            Folder: <strong><?= htmlspecialchars($folder['nama_folder']) ?></strong> | 
            Kategori: <strong><?= htmlspecialchars($folder['nama_kategori']) ?></strong>
            (<?= $folder['tipe'] == 'dinamis' ? 'Data Dinamis' : 'Data Statis' ?>)
        </p>
        <hr>
    </div>
</div>

<div class="row">
    <div class="col-md-8 mx-auto">
        <div class="card">
            <div class="card-header bg-primary text-white">
                <h5 class="card-title mb-0"><i class="fas fa-file-upload"></i> Form Upload Dokumen <?= $isReplace ? '(Unggah Ulang Draft)' : '' ?></h5>
            </div>
            <div class="card-body">
                <form method="post" action="<?= BASE_URL ?>/pages/upload?folder_id=<?= (int)$folder_id ?><?= $isReplace ? '&replace=1' : '' ?>" enctype="multipart/form-data">
                    <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
                    <?php
                    // Prefill jika mode replace
                    $prefJudul = '';
                    // Deskripsi tidak digunakan lagi
                    if ($isReplace) {
                        $stmtPref = $conn->prepare("SELECT judul FROM dokumen WHERE folder_id = ? AND user_id = ? AND status = 'draft' ORDER BY id DESC LIMIT 1");
                        $stmtPref->bind_param("ii", $folder_id, $_SESSION['user_id']);
                        $stmtPref->execute();
                        $resPref = $stmtPref->get_result();
                        if ($resPref && $resPref->num_rows) {
                            $rowPref = $resPref->fetch_assoc();
                            $prefJudul = $rowPref['judul'] ?? '';
                        }
                        $stmtPref->close();
                        echo '<div class="alert alert-info"><i class="fas fa-info-circle"></i> Mode unggah ulang: dokumen draft Anda akan diperbarui dan diajukan kembali untuk verifikasi.</div>';
                    }
                    ?>
                    <?php if (hasRole('administrator') || hasRole('root')): ?>
                    <div class="mb-3">
                        <label for="owner_user_id" class="form-label">Pemilik Dokumen</label>
                        <select class="form-select" id="owner_user_id" name="owner_user_id">
                            <option value="<?= (int)$_SESSION['user_id'] ?>"><?= htmlspecialchars($_SESSION['user_name']) ?> (Saya)</option>
                            <?php
                                $resOwners = $conn->query("SELECT id, nama_lengkap, role FROM users WHERE role = 'pendidik' AND status = 1 ORDER BY nama_lengkap ASC");
                                if ($resOwners) {
                                    while($u = $resOwners->fetch_assoc()) {
                                        echo '<option value="' . (int)$u['id'] . '">' . htmlspecialchars($u['nama_lengkap']) . ' (' . ucfirst($u['role']) . ")</option>";
                                    }
                                }
                            ?>
                        </select>
                        <div class="form-text">Admin/Root dapat mengunggah atas nama pendidik.</div>
                    </div>
                    <?php endif; ?>
                    <!-- Input judul dihapus: judul akan diambil dari nama file PDF -->
                    <!-- Deskripsi dihapus -->
                    <div class="mb-3">
                        <label for="dokumen" class="form-label">File Dokumen</label>
                        <input type="file" class="form-control" id="dokumen" name="dokumen" accept="application/pdf" required>
                        <div class="form-text">Ukuran maksimal: <?= MAX_FILE_SIZE / 1024 / 1024 ?>MB. Hanya PDF.</div>
                    </div>
                    <div class="mb-3">
                        <div id="preview" class="mt-2 d-none">
                            <h6>Preview:</h6>
                            <div id="preview-content" class="border p-2 rounded"></div>
                        </div>
                    </div>
                    <div class="d-grid gap-2">
                        <button type="submit" name="upload" class="btn btn-primary">
                            <i class="fas fa-upload"></i> Upload Dokumen
                        </button>
        <a href="<?= BASE_URL ?>/pages/documents?folder_id=<?= $folder_id ?>" class="btn btn-secondary">
            <i class="fas fa-arrow-left"></i> Kembali
        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<?php
include '../includes/footer.php';
?>