<?php
require_once '../config/config.php';
require_once '../config/database.php';

// Cek login dan role (root dan administrator diizinkan)
if (!isLoggedIn() || !(hasRole('administrator') || hasRole('root'))) {
redirect('/login');
}

$error = '';
$success = '';

// Cek ID user
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
redirect('/pages/users');
}

$user_id = $_GET['id'];

// Ambil data user
$stmt = $conn->prepare("SELECT id, username, nama_lengkap, email, role, status FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows == 0) {
redirect('/pages/users');
}

$user = $result->fetch_assoc();

// Proses edit user
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Validasi CSRF
    $token = $_POST['csrf_token'] ?? '';
    if (!verify_csrf($token)) {
        $error = 'Sesi tidak valid. Silakan muat ulang halaman dan coba lagi.';
    } else {
    $nama_lengkap = sanitize($_POST['nama_lengkap']);
    $email = sanitize($_POST['email']);
    $role = sanitize($_POST['role']);
    $status = isset($_POST['status']) ? 1 : 0;
    $password = $_POST['password'];
    $currentRole = $_SESSION['user_role'] ?? '';
    // Jika admin mengedit akun admin/root, paksa role tetap seperti semula
    if ($currentRole === 'administrator' && in_array($user['role'], ['administrator','root'])) {
        $role = $user['role'];
    }
    
    // Validasi input
    if (empty($nama_lengkap) || empty($email) || empty($role)) {
        $error = 'Nama, email, dan role harus diisi';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Format email tidak valid';
    } elseif (!in_array($role, ['root', 'administrator', 'verifikator', 'pendidik'])) {
        $error = 'Role tidak valid';
    } elseif ($currentRole === 'administrator' && in_array($role, ['administrator','root']) && $role !== $user['role']) {
        $error = 'Administrator tidak dapat mengubah role menjadi administrator atau root';
    } else {
        // Cek email sudah digunakan oleh user lain atau belum
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
        $stmt->bind_param("si", $email, $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $error = 'Email sudah digunakan oleh pengguna lain';
        } else {
            // Update user
            if (!empty($password)) {
                // Update dengan password baru
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $conn->prepare("UPDATE users SET nama_lengkap = ?, email = ?, role = ?, status = ?, password = ? WHERE id = ?");
                $stmt->bind_param("sssisi", $nama_lengkap, $email, $role, $status, $hashed_password, $user_id);
            } else {
                // Update tanpa password
                $stmt = $conn->prepare("UPDATE users SET nama_lengkap = ?, email = ?, role = ?, status = ? WHERE id = ?");
                $stmt->bind_param("sssii", $nama_lengkap, $email, $role, $status, $user_id);
            }
            
            if ($stmt->execute()) {
                $success = 'Data pengguna berhasil diperbarui';
                
                // Log aktivitas
                $admin_id = $_SESSION['user_id'];
                $aktivitas = "Memperbarui data pengguna: " . $user['username'];
                $ip = $_SERVER['REMOTE_ADDR'];
                $stmt = $conn->prepare("INSERT INTO log_aktivitas (user_id, aktivitas, ip_address) VALUES (?, ?, ?)");
                $stmt->bind_param("iss", $admin_id, $aktivitas, $ip);
                $stmt->execute();
                
                // Refresh data user
                $stmt = $conn->prepare("SELECT id, username, nama_lengkap, email, role, status FROM users WHERE id = ?");
                $stmt->bind_param("i", $user_id);
                $stmt->execute();
                $result = $stmt->get_result();
                $user = $result->fetch_assoc();
            } else {
                $error = 'Gagal memperbarui data pengguna: ' . $conn->error;
            }
        }
    }
}
}
// Include header
$pageTitle = "Edit Pengguna";
include '../includes/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Edit Pengguna</h2>
<a href="<?= BASE_URL ?>/pages/users" class="btn btn-secondary">
        <i class="fas fa-arrow-left"></i> Kembali
    </a>
</div>

<div class="card">
    <div class="card-body">
        <?php if (!empty($error)): ?>
            <div class="alert alert-danger"><?= htmlspecialchars($error, ENT_QUOTES, 'UTF-8') ?></div>
        <?php endif; ?>
        
        <?php if (!empty($success)): ?>
            <div class="alert alert-success"><?= htmlspecialchars($success, ENT_QUOTES, 'UTF-8') ?></div>
        <?php endif; ?>
        
        <form method="POST" action="">
            <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="username" class="form-label">Username</label>
                    <input type="text" class="form-control" id="username" value="<?= htmlspecialchars($user['username'], ENT_QUOTES, 'UTF-8') ?>" readonly disabled>
                    <small class="text-muted">Username tidak dapat diubah</small>
                </div>
                <div class="col-md-6">
                    <label for="password" class="form-label">Password</label>
                    <input type="password" class="form-control" id="password" name="password">
                    <small class="text-muted">Kosongkan jika tidak ingin mengubah password</small>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="nama_lengkap" class="form-label">Nama Lengkap</label>
                    <input type="text" class="form-control" id="nama_lengkap" name="nama_lengkap" value="<?= htmlspecialchars($user['nama_lengkap'], ENT_QUOTES, 'UTF-8') ?>" required>
                </div>
                <div class="col-md-6">
                    <label for="email" class="form-label">Email</label>
                    <input type="email" class="form-control" id="email" name="email" value="<?= htmlspecialchars($user['email'], ENT_QUOTES, 'UTF-8') ?>" required>
                </div>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="role" class="form-label">Role</label>
                    <?php $currentRole = $_SESSION['user_role'] ?? ''; ?>
                    <?php if ($currentRole === 'administrator' && in_array($user['role'], ['administrator','root'])): ?>
                        <select class="form-select" id="role" name="role" disabled>
                            <option value="<?= $user['role'] ?>" selected><?= $user['role'] === 'root' ? 'Root (Super Administrator)' : 'Administrator' ?></option>
                        </select>
                        <small class="text-muted">Administrator tidak dapat mengubah role akun ini</small>
                        <input type="hidden" name="role" value="<?= htmlspecialchars($user['role'], ENT_QUOTES, 'UTF-8') ?>">
                    <?php else: ?>
                        <select class="form-select" id="role" name="role" required>
                            <?php if ($currentRole === 'root'): ?>
                                <option value="administrator" <?= $user['role'] == 'administrator' ? 'selected' : '' ?>>Administrator</option>
                                <option value="verifikator" <?= $user['role'] == 'verifikator' ? 'selected' : '' ?>>Verifikator</option>
                                <option value="pendidik" <?= $user['role'] == 'pendidik' ? 'selected' : '' ?>>Pendidik</option>
                            <?php else: ?>
                                <option value="verifikator" <?= $user['role'] == 'verifikator' ? 'selected' : '' ?>>Verifikator</option>
                                <option value="pendidik" <?= $user['role'] == 'pendidik' ? 'selected' : '' ?>>Pendidik</option>
                            <?php endif; ?>
                        </select>
                    <?php endif; ?>
                </div>
                <div class="col-md-6">
                    <div class="form-check form-switch mt-4">
                        <input class="form-check-input" type="checkbox" id="status" name="status" <?= $user['status'] == 1 ? 'checked' : '' ?>>
                        <label class="form-check-label" for="status">Akun Aktif</label>
                    </div>
                </div>
            </div>
            
            <div class="d-grid gap-2">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i> Simpan Perubahan
                </button>
            </div>
        </form>
    </div>
</div>

<?php include '../includes/footer.php'; ?>