<?php
require_once '../config/config.php';
require_once '../config/database.php';

// Cek login dan role (administrator dan root diizinkan)
if (!isLoggedIn() || !(hasRole('administrator') || hasRole('root'))) {
redirect('/login');
}


// Proses hapus user (administrator tidak boleh menghapus root) dengan validasi CSRF
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $token = $_GET['csrf'] ?? '';
    if (!verify_csrf($token)) {
        $_SESSION['alert'] = "Sesi tidak valid. Silakan muat ulang halaman.";
        $_SESSION['alert_type'] = "danger";
        redirect('/pages/users');
    }
    $id = (int)$_GET['delete'];
    // Cek apakah user ada dan ambil role/username
    $stmt = $conn->prepare("SELECT id, role, username FROM users WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result && $result->num_rows > 0) {
        $target = $result->fetch_assoc();
        if ($target['role'] === 'root' || $target['username'] === 'root') {
            $_SESSION['alert'] = "Administrator tidak boleh menghapus akun root";
            $_SESSION['alert_type'] = "danger";
        } else {
            $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
            $stmt->bind_param("i", $id);
            if ($stmt->execute()) {
                $_SESSION['alert'] = "Pengguna berhasil dihapus";
                $_SESSION['alert_type'] = "success";
            } else {
                $_SESSION['alert'] = "Gagal menghapus pengguna: " . $conn->error;
                $_SESSION['alert_type'] = "danger";
            }
        }
    } else {
        $_SESSION['alert'] = "Pengguna tidak ditemukan";
        $_SESSION['alert_type'] = "danger";
    }
redirect('/pages/users');
}

// Proses ubah status user (admin tidak boleh mengubah status akun root) dengan validasi CSRF
if (isset($_GET['toggle']) && is_numeric($_GET['toggle'])) {
    $tok = $_GET['csrf'] ?? '';
    if (!verify_csrf($tok)) {
        $_SESSION['alert'] = "Sesi tidak valid. Silakan muat ulang halaman.";
        $_SESSION['alert_type'] = "danger";
        redirect('/pages/users');
    }
    $id = $_GET['toggle'];
    $currentRole = $_SESSION['user_role'] ?? '';
    
    // Cek apakah user ada dan ambil status saat ini
    $stmt = $conn->prepare("SELECT id, status, role, username FROM users WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $user = $result->fetch_assoc();
        if ($currentRole === 'administrator' && ($user['role'] === 'root' || $user['username'] === 'root')) {
            $_SESSION['alert'] = "Administrator tidak boleh mengubah status akun root";
            $_SESSION['alert_type'] = "danger";
redirect('/pages/users');
        }
        $newStatus = $user['status'] == 1 ? 0 : 1;
        
        // Update status
        $stmt = $conn->prepare("UPDATE users SET status = ? WHERE id = ?");
        $stmt->bind_param("ii", $newStatus, $id);
        
        if ($stmt->execute()) {
            $_SESSION['alert'] = "Status pengguna berhasil diubah";
            $_SESSION['alert_type'] = "success";
        } else {
            $_SESSION['alert'] = "Gagal mengubah status pengguna: " . $conn->error;
            $_SESSION['alert_type'] = "danger";
        }
    } else {
        $_SESSION['alert'] = "Pengguna tidak ditemukan";
        $_SESSION['alert_type'] = "danger";
    }
    
redirect('/pages/users');
}

// Pencarian dan pagination
$q = isset($_GET['q']) ? trim($_GET['q']) : '';
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$perPage = 15;
$offset = ($page - 1) * $perPage;

// Bangun WHERE sesuai role dan pencarian
$whereParts = [];
$params = [];
$types = '';
$currentRole = $_SESSION['user_role'] ?? '';
if ($currentRole === 'administrator') {
    $whereParts[] = "(role IS NULL OR role != 'root')";
    $whereParts[] = "username != 'root'";
}
if ($q !== '') {
    $whereParts[] = "(username LIKE ? OR nama_lengkap LIKE ?)";
    $params[] = "%$q%";
    $params[] = "%$q%";
    $types .= 'ss';
}
$whereSql = count($whereParts) ? (' WHERE ' . implode(' AND ', $whereParts)) : '';

// Hitung total baris
$countSql = "SELECT COUNT(*) AS total FROM users" . $whereSql;
$stmt = $conn->prepare($countSql);
if ($types !== '') { $stmt->bind_param($types, ...$params); }
$stmt->execute();
$res = $stmt->get_result();
$totalRows = (int)($res->fetch_assoc()['total'] ?? 0);
$stmt->close();
$totalPages = max(1, (int)ceil($totalRows / $perPage));
if ($page > $totalPages) { $page = $totalPages; $offset = ($page - 1) * $perPage; }

// Ambil data halaman (tambahkan IP terakhir login dari log_aktivitas)
$dataSql = "SELECT id, username, nama_lengkap, role, status, created_at, last_login,
                   (SELECT ip_address FROM log_aktivitas la WHERE la.user_id = users.id AND la.aktivitas LIKE 'Login%' ORDER BY la.created_at DESC LIMIT 1) AS last_ip
            FROM users" . $whereSql . " ORDER BY nama_lengkap ASC LIMIT ? OFFSET ?";
$stmt = $conn->prepare($dataSql);
if ($types !== '') {
    $bindTypes = $types . 'ii';
    $stmt->bind_param($bindTypes, ...array_merge($params, [$perPage, $offset]));
} else {
    $stmt->bind_param('ii', $perPage, $offset);
}
$stmt->execute();
$result = $stmt->get_result();
$users = $result->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Include header
$pageTitle = "Manajemen Pengguna";
include '../includes/header.php';
?>

<div class="row align-items-center mb-3">
    <div class="col-md-7">
        <h2 class="mb-0"><i class="fas fa-users"></i> Manajemen Pengguna</h2>
    </div>
    <div class="col-md-5 d-flex justify-content-end">
        <div class="btn-group flex-nowrap">
            <a href="<?= BASE_URL ?>/pages/add_user" class="btn btn-primary d-inline-flex align-items-center gap-2 text-nowrap">
                <i class="fas fa-plus-circle fa-fw"></i> Tambah Pengguna
            </a>
            <a href="<?= BASE_URL ?>/pages/import_users" class="btn btn-outline-primary d-inline-flex align-items-center gap-2 text-nowrap">
                <i class="fas fa-file-upload fa-fw"></i> Impor CSV
            </a>
            <a href="<?= BASE_URL ?>/assets/templates/user_import_template.csv" class="btn btn-outline-secondary d-inline-flex align-items-center gap-2 text-nowrap">
                <i class="fas fa-download fa-fw"></i> Download CSV
            </a>
        </div>
    </div>
</div>

<!-- Form pencarian diselaraskan dengan gaya Data Statis/Dinamis -->
<div class="row mb-3">
    <div class="col-12">
        <form method="get" action="<?= BASE_URL ?>/pages/users" class="d-flex gap-2">
            <input type="text" name="q" value="<?= htmlspecialchars($q) ?>" class="form-control" placeholder="Cari Nama atau Username">
            <button type="submit" class="btn btn-primary"><i class="fas fa-search"></i></button>
        </form>
    </div>
</div>

<div class="card">
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                    <tr>
                        <th>No</th>
                        <th>Username</th>
                        <th>Nama Lengkap</th>
                        <th>Role</th>
                        <th>Status</th>
                        <th>Terakhir Login</th>
                        <th>IP Address</th>
                        <th>Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (count($users) > 0): ?>
                        <?php $no = 1; foreach ($users as $user): ?>
                            <?php 
                                $currentRole = $_SESSION['user_role'] ?? ''; 
                                // Sembunyikan akun root bagi administrator (baik role 'root' maupun username 'root')
                                if ($currentRole === 'administrator' && ($user['role'] === 'root' || $user['username'] === 'root')) continue; 
                            ?>
                            <tr>
                                <td><?= $no++ ?></td>
                                <td><?= htmlspecialchars($user['username'], ENT_QUOTES, 'UTF-8') ?></td>
                                <td><?= htmlspecialchars($user['nama_lengkap'], ENT_QUOTES, 'UTF-8') ?></td>
                                <td>
                                    <span class="badge bg-<?=
                                        $user['role'] == 'root' ? 'dark' : (
                                            $user['role'] == 'administrator' ? 'danger' : (
                                                $user['role'] == 'verifikator' ? 'warning' : 'info'
                                            )
                                        )
                                    ?>">
                                        <?= $user['role'] == 'root' ? 'Root' : ucfirst($user['role']) ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-<?= $user['status'] == 1 ? 'success' : 'secondary' ?>">
                                        <?= $user['status'] == 1 ? 'Aktif' : 'Nonaktif' ?>
                                    </span>
                                </td>
                                <td><?= !empty($user['last_login']) ? date('d/m/Y H:i', strtotime($user['last_login'])) : '-' ?></td>
                                <td><?= !empty($user['last_ip']) ? htmlspecialchars($user['last_ip']) : '-' ?></td>
                                <td>
                                    <div class="btn-group">
                                        <?php $isRootRow = ($user['role'] === 'root' || $user['username'] === 'root'); ?>
                                        <?php $isAdminRow = ($user['role'] === 'administrator'); ?>
                                        <?php
                                            // Tombol Edit: nonaktif jika baris root
                    $editHref = $isRootRow ? 'javascript:void(0)' : (BASE_URL . '/pages/edit_user?id=' . $user['id']);
                                            $editClass = 'btn btn-sm btn-info' . ($isRootRow ? ' disabled' : '');
                                        ?>
                                        <a href="<?= $editHref ?>" class="<?= $editClass ?>" <?= $isRootRow ? 'tabindex="-1" aria-disabled="true"' : '' ?> title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <?php
                                            // Tombol Nonaktif/Aktif: nonaktif jika baris root
                                            $toggleHref = ($isRootRow || $isAdminRow) ? 'javascript:void(0)' : (BASE_URL . '/pages/users?toggle=' . $user['id'] . '&csrf=' . csrf_token());
                                            $toggleClass = 'btn btn-sm btn-' . ($user['status'] == 1 ? 'warning' : 'success') . (($isRootRow || $isAdminRow) ? ' disabled' : '');
                                            $toggleMsg = 'Apakah Anda yakin ingin ' . ($user['status'] == 1 ? 'menonaktifkan' : 'mengaktifkan') . ' pengguna ini?';
                                        ?>
                                        <a href="<?= $toggleHref ?>" class="<?= $toggleClass ?> btn-toggle" data-confirm-message="<?= htmlspecialchars($toggleMsg, ENT_QUOTES, 'UTF-8') ?>" <?= $isRootRow ? 'tabindex="-1" aria-disabled="true"' : '' ?> title="Nonaktif/Aktif">
                                            <i class="fas fa-<?= $user['status'] == 1 ? 'ban' : 'check' ?>"></i>
                                        </a>
                                        <?php if ($isRootRow): ?>
                                            <a href="javascript:void(0)" class="btn btn-sm btn-danger disabled" tabindex="-1" aria-disabled="true" title="Hapus">
                                                <i class="fas fa-trash"></i>
                                            </a>
                                        <?php elseif ($isAdminRow): ?>
                                            <a href="javascript:void(0)" class="btn btn-sm btn-danger disabled" tabindex="-1" aria-disabled="true" title="Hapus">
                                                <i class="fas fa-trash"></i>
                                            </a>
                                        <?php else: ?>
                                            <a href="<?= BASE_URL ?>/pages/users?delete=<?= $user['id'] ?>&csrf=<?= csrf_token() ?>" class="btn btn-sm btn-danger btn-delete" 
                                               data-confirm-message="Apakah Anda yakin ingin menghapus pengguna ini? Semua dokumen yang terkait juga akan dihapus." title="Hapus">
                                                <i class="fas fa-trash"></i>
                                            </a>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="8" class="text-center">Tidak ada pengguna</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<?php if ($totalPages > 1): ?>
<?php
  $base = BASE_URL . '/pages/users?';
  $qs = [];
  if ($q !== '') { $qs['q'] = $q; }
  $build = function($p) use ($base, $qs) {
      $qs['page'] = $p;
      return $base . http_build_query($qs);
  };
  echo render_pagination($page, $totalPages, $build, 'Navigasi halaman');
?>
<?php endif; ?>

<?php include '../includes/footer.php'; ?>